/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpParseCommandLine.c,v 1.6 2001/02/12 19:43:52 sluu Exp $
____________________________________________________________________________*/
#include "pgpPFLConfig.h"
#include "pgpPFLPriv.h"
#include "pgpParseCommandLine.h"

#include "pgpBase.h"
#ifndef NO_SDK_LIB
#include "pgpMemoryMgr.h"
#endif /* NO_SDK_LIB */
#include "pgpMem.h"
#include "pgpPFLErrors.h"


#define	CKERR			if( IsPGPError( err ) ) { goto done; }
#define CKNULL(val)		if( IsNull( val ) ) \
						{ \
							err = kPGPError_OutOfMemory;\
							goto done;\
						}
#define RETERR(reterr)	{ \
							err = (reterr);\
							goto done;\
						}

typedef struct PGPCommandLineContext
{
#ifndef NO_SDK_LIB
	PGPMemoryMgrRef			mgr;
#endif /* NO_SDK_LIB */

	PGPUInt32				argc;
	char **					argv;

	PGPCommandLineLongArg *	longArgs;
	PGPSize					numLongArgs;

	PGPUInt32				nextFlagIndex;		/* Index of current argv element */
	PGPUInt32				nextFlagArgvIndex;	/* Index into current argv element */

	PGPBoolean *			argUsed;			/* Array of used flags for args */
} PGPCommandLineContext;


	PGPError
PGPNewCommandLineContext(
#ifndef NO_SDK_LIB
	PGPMemoryMgrRef				mgr,
#endif /* NO_SDK_LIB */
	PGPUInt32					argc,
	char *						argv[],
	PGPCommandLineLongArg *		longArgs,
	PGPSize						numLongArgs,
	PGPCommandLineContextRef *	clContext )
{
	PGPCommandLineContext *		con		= NULL;
	PGPUInt32					i		= 0;
	PGPError					err		= kPGPError_NoErr;

#ifdef NO_SDK_LIB
	con = calloc(1, sizeof(PGPCommandLineContext));
#else /* NO_SDK_LIB */
	PGPValidatePtr( mgr );

	con = PGPNewData( mgr,
		sizeof( PGPCommandLineContext ),
		kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
	CKNULL( con );

#ifndef NO_SDK_LIB
	con->mgr		= mgr;
#endif /* NO_SDK_LIB */

	/* Copy argc and argv */
	con->argc = argc;
#ifdef NO_SDK_LIB
	con->argv = calloc(argc, sizeof(char *));
#else /* NO_SDK_LIB */
	con->argv = PGPNewData( con->mgr, argc * sizeof( char * ), kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
	CKNULL( con->argv );

	for( i = 0; i < argc; i++ )
	{
		if( IsNull( argv[i] ) )
			RETERR( kPGPError_BadParams );

#ifdef NO_SDK_LIB
		con->argv[i] = calloc(strlen(argv[i]) + 1, sizeof(char));
#else /* NO_SDK_LIB */
		con->argv[i] = PGPNewData( con->mgr,
			strlen( argv[i] ) + 1,
			kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
		CKNULL( con->argv[i] );

		pgpCopyMemory( argv[i], con->argv[i], strlen( argv[i] ) + 1 );
	}

	/* Copy longArgs and numLongArgs */
	if( ( numLongArgs > 0 ) && IsntNull( longArgs ) )
	{
		con->numLongArgs = numLongArgs;
#ifdef NO_SDK_LIB
		con->longArgs = calloc(numLongArgs, sizeof(PGPCommandLineLongArg));
#else /* NO_SDK_LIB */
		con->longArgs = PGPNewData( con->mgr,
			numLongArgs * sizeof( PGPCommandLineLongArg ),
			kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
		CKNULL( con->longArgs );

		for( i = 0; i < numLongArgs; i++ )
		{
			if( IsNull( longArgs[i].argString ) )
				RETERR( kPGPError_BadParams );

#ifdef NO_SDK_LIB
			con->longArgs[i].argString = calloc(strlen( longArgs[i].argString ) + 1, sizeof(char));
#else /* NO_SDK_LIB */
			con->longArgs[i].argString = PGPNewData( con->mgr,
				strlen( longArgs[i].argString ) + 1,
				kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
			CKNULL( con->longArgs[i].argString );

			pgpCopyMemory( longArgs[i].argString,
				con->longArgs[i].argString,
				strlen( longArgs[i].argString ) + 1 );

			con->longArgs[i].value = longArgs[i].value;
		}
	}
	else
	{
		con->numLongArgs	= 0;
		con->longArgs		= NULL;
	}

#ifdef NO_SDK_LIB
	con->argUsed = calloc(argc, sizeof(PGPBoolean));
#else /* NO_SDK_LIB */
	con->argUsed = PGPNewData( con->mgr,
		argc * sizeof( PGPBoolean ),
		kPGPMemoryMgrFlags_Clear );
#endif /* NO_SDK_LIB */
	CKNULL( con->argUsed );

	for( i = 0; i < argc; i++ )
		con->argUsed[i] = FALSE;

	/* Indexes - be sure to skip argv[0] as it contains the program name */
	con->nextFlagIndex			= 1;
	con->nextFlagArgvIndex		= 0;

	*clContext = con;

done:
	if( IsPGPError( err ) )
		(void) PGPFreeCommandLineContext( con );

	return err;
}

	PGPError
PGPFreeCommandLineContext(
	PGPCommandLineContextRef	con )
{
	PGPUInt32		i			= 0;
	PGPError		err			= kPGPError_NoErr;

#ifndef NO_SDK_LIB
	PGPValidatePtr( con );
#endif /* NO_SDK_LIB */

	if( IsntNull( con->argv ) )
	{
		for( i = 0; i < con->argc; i++ )
		{
			if( IsntNull( con->argv[i] ) )
				(void) PGPFreeData( con->argv[i] );
		}
		(void) PGPFreeData( con->argv );
	}

	if( IsntNull( con->longArgs ) )
	{
		for( i = 0; i < con->numLongArgs; i++ )
		{
			if( IsntNull( con->longArgs[i].argString ) )
				(void) PGPFreeData( con->longArgs[i].argString );
		}
		(void) PGPFreeData( con->longArgs );
	}

	if( IsntNull( con->argUsed ) )
		(void) PGPFreeData( con->argUsed );

	(void) PGPFreeData( con );

	return err;
}

	PGPError
PGPCommandLineNextFlag(
	PGPCommandLineContextRef	con,
	PGPUInt32 *					flag )
{
	PGPUInt32		i			= 0;
	PGPUInt32		nextFlag	= 0;
	PGPError		err			= kPGPError_NoErr;

#ifndef NO_SDK_LIB
	PGPValidatePtr( con );
	PGPValidatePtr( flag );
#endif /* NO_SDK_LIB */

	if( con->nextFlagIndex >= con->argc )
		RETERR( kPGPError_EndOfIteration );

	/* End of current argv element.  Move on to next one */
	if( con->nextFlagArgvIndex >= strlen( con->argv[con->nextFlagIndex] ) )
	{
		con->nextFlagIndex++;
		con->nextFlagArgvIndex = 0;
		if( con->nextFlagIndex == con->argc )
			RETERR( kPGPError_EndOfIteration );
	}

	/* Skip all the arguments */
	while( ( con->nextFlagIndex < con->argc )
		&& ( ( con->argv[con->nextFlagIndex][0] != '-' )
			|| ( strlen( con->argv[con->nextFlagIndex] ) == 1 ) ) )
	{
		con->nextFlagIndex++;
	}

	if( con->nextFlagIndex == con->argc )
		RETERR( kPGPError_EndOfIteration );

	/* Next char in current argv element */
	if( con->nextFlagArgvIndex < strlen( con->argv[con->nextFlagIndex] ) )
	{
		/* Skip all the hyphens */
		while( ( con->nextFlagIndex < con->argc )
			&& ( con->argv[con->nextFlagIndex][con->nextFlagArgvIndex] == '-' )
			&& ( con->argv[con->nextFlagIndex][con->nextFlagArgvIndex] != '\0' ) )
		{
			if( strncmp( con->argv[con->nextFlagIndex], "--", 2 ) == 0 )
			{
				/* We found a long argument */
				for( i = 0; i < con->numLongArgs; i++ )
				{
					if( strcmp( con->argv[con->nextFlagIndex] + 2, con->longArgs[i].argString ) == 0 )
					{
						nextFlag = con->longArgs[i].value;
						break;
					}
				}
				con->nextFlagIndex++;
				con->nextFlagArgvIndex = 0;
				if( nextFlag )
					goto done;
			}
			con->nextFlagArgvIndex++;
		}

		if( con->nextFlagIndex == con->argc )
			RETERR( kPGPError_EndOfIteration );

		if( con->argv[con->nextFlagIndex][con->nextFlagArgvIndex] == '\0' )
		{
			/* We reached the end of the current argv element.  Move on to the next. */
			err = PGPCommandLineNextFlag( con, flag );
			goto done;
		}

		nextFlag = con->argv[con->nextFlagIndex][con->nextFlagArgvIndex];
		con->nextFlagArgvIndex++;
		goto done;
	}

done:
	if( IsntPGPError( err ) && nextFlag )
		*flag = nextFlag;

	return err;
}

	PGPError
PGPCommandLineNextArgument(
	PGPCommandLineContextRef	con,
	char *						arg,
	PGPSize						allocLen,
	PGPSize *					outLen )
{
	PGPError		err				= kPGPError_NoErr;
	PGPUInt32		nextArgIndex	= 0;

#ifndef NO_SDK_LIB
	PGPValidatePtr( con );
#endif /* NO_SDK_LIB */

	if( IsntNull( outLen ) )
		*outLen = 0;

	if( con->nextFlagIndex >= con->argc )
		RETERR( kPGPError_EndOfIteration );

	/* End of current argv element.  Move on to next one */
	if( con->nextFlagArgvIndex >= strlen( con->argv[con->nextFlagIndex] ) )
	{
		con->nextFlagIndex++;
		con->nextFlagArgvIndex = 0;
	}
	nextArgIndex = con->nextFlagIndex;

	while( nextArgIndex < con->argc )
	{
		if( con->argUsed[nextArgIndex] == FALSE )
		{
			if( con->argv[nextArgIndex][0] == '-' )
			{
				/* Found either a flag or the string "-" */
				if( strlen( con->argv[nextArgIndex] ) == 1 )
				{
					/* string "-" */
					break;
				}
				else
				{
					/* flag */
					RETERR( kPGPError_ItemNotFound );
				}
			}
			else
				break;
		}

		nextArgIndex++;
	}
	if( nextArgIndex >= con->argc )
		RETERR( kPGPError_EndOfIteration );

	if( IsntNull( outLen ) )
		*outLen = strlen( con->argv[nextArgIndex] );

	if( allocLen < strlen( con->argv[nextArgIndex] ) + 1 )
		RETERR( kPGPError_BufferTooSmall );

	if( IsntNull( arg ) )
	{
		pgpCopyMemory( con->argv[nextArgIndex],
			arg,
			strlen( con->argv[nextArgIndex] ) + 1 );

		con->argUsed[nextArgIndex] = TRUE;
	}

done:
	return err;
}

	PGPError
PGPCommandLineGetFollowingText(
	PGPCommandLineContextRef	con,
	char *						arg,
	PGPSize						allocLen,
	PGPSize *					outLen )
{
	PGPError		err			= kPGPError_NoErr;

#ifndef NO_SDK_LIB
	PGPValidatePtr( con );
#endif /* NO_SDK_LIB */

	if( con->nextFlagIndex >= con->argc )
		RETERR( kPGPError_EndOfIteration );

	if( IsntNull( outLen ) )
		*outLen = strlen( con->argv[con->nextFlagIndex] + con->nextFlagArgvIndex );

	if( allocLen < strlen( con->argv[con->nextFlagIndex] + con->nextFlagArgvIndex ) + 1 )
		RETERR( kPGPError_BufferTooSmall );

	if( IsntNull( arg ) )
	{
		pgpCopyMemory( con->argv[con->nextFlagIndex] + con->nextFlagArgvIndex,
			arg,
			strlen( con->argv[con->nextFlagIndex] + con->nextFlagArgvIndex ) + 1 );

		con->nextFlagIndex++;
		con->nextFlagArgvIndex = 0;
	}

done:
	return err;
}




/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
